package dev.scsupercraft.mc.libraries.corelib.api.util;

import dev.architectury.event.events.common.PlayerEvent;
import dev.architectury.networking.NetworkManager;
import dev.scsupercraft.mc.libraries.corelib.CoreLib;
import dev.scsupercraft.mc.libraries.corelib.api.data.SyncedData;
import dev.scsupercraft.mc.libraries.corelib.api.serialization.CodecHolder;
import org.jetbrains.annotations.ApiStatus;

import java.util.HashMap;
import java.util.Map;
import net.minecraft.class_2540;
import net.minecraft.class_2960;
import net.minecraft.class_3222;
import net.minecraft.class_8710;
import net.minecraft.class_9139;

/**
 * A utility class for synchronising data.
 */
@ApiStatus.AvailableSince("1.0.0")
public class Synchronisation {
	private static final Map<class_2960, SyncedObject<?>> SYNCED_OBJECTS = new HashMap<>();

	static {
		NetworkManager.registerReceiver(NetworkManager.Side.S2C, DataSyncPacket.PACKET_ID, DataSyncPacket.PACKET_CODEC, (packet, context) -> {
			SyncedData<Object> syncedData = Synchronisation.get(packet.id()).syncedData();
			syncedData.setSyncedData(packet.data());
			CoreLib.LOGGER.info("Synchronised {} with the server.", packet.id());
		});

		PlayerEvent.PLAYER_JOIN.register(Synchronisation::synchronise);
	}

	/**
	 * Set up an object for synchronisation.
	 * <p>
	 * Failing to call this method will result in errors upon trying to synchronise the data.
	 * @param syncedData  The synced data to set up synchronisation for.
	 * @param codecHolder The codec holder to use for serialisation.
	 * @param <T>         The type of data being synchronised.
	 */
	@ApiStatus.AvailableSince("1.0.0")
	public static <T> void setup(SyncedData<T> syncedData, CodecHolder<T> codecHolder) {
		if (!SYNCED_OBJECTS.containsKey(syncedData.getSyncId())) SYNCED_OBJECTS.put(syncedData.getSyncId(), new SyncedObject<>(syncedData, codecHolder));
	}

	/**
	 * Synchronises the data to all clients.
	 * @param syncedData The data to synchronise.
	 */
	@ApiStatus.AvailableSince("1.0.0")
	public static void synchronise(SyncedData<?> syncedData) {
		synchronise(SYNCED_OBJECTS.get(syncedData.getSyncId()));
	}

	/**
	 * Synchronises the data to the specified client.
	 * @param syncedData The data to synchronise.
	 * @param client     The client to send the data to.
	 */
	@ApiStatus.AvailableSince("1.0.0")
	public static void synchronise(SyncedData<?> syncedData, class_3222 client) {
		synchronise(SYNCED_OBJECTS.get(syncedData.getSyncId()), client);
	}

	/**
	 * Synchronises all data to the specified client.
	 * @param client The client to send the data to.
	 */
	@ApiStatus.AvailableSince("1.0.0")
	public static void synchronise(class_3222 client) {
		SYNCED_OBJECTS.values().forEach(obj -> synchronise(obj, client));
	}

	private static void synchronise(SyncedObject<?> syncedObject) {
		CoreLib.LOGGER.info("Synchronising {} to all clients...", syncedObject.syncedData.getSyncId());
		syncedObject.sync();
	}
	private static void synchronise(SyncedObject<?> syncedObject, class_3222 client) {
		CoreLib.LOGGER.info("Synchronising {} to client {}...", syncedObject.syncedData.getSyncId(), client.method_7334().getName());
		syncedObject.sync(client);
	}

	private static <T> SyncedObject<T> get(class_2960 id) {
		return Utils.cast(SYNCED_OBJECTS.get(id));
	}

	private record SyncedObject<T>(SyncedData<T> syncedData, CodecHolder<T> codecHolder) {
		private DataSyncPacket<T> createPacket() {
			return new DataSyncPacket<>(syncedData.getSyncId(), syncedData.getData());
		}

		private void sync() {
			if (CoreLib.server == null) return;
			NetworkManager.sendToPlayers(CoreLib.server.method_3760().method_14571(), createPacket());
		}

		private void sync(class_3222 client) {
			if (CoreLib.server == null || client == null) return;
			NetworkManager.sendToPlayer(client, createPacket());
		}
	}
	private record DataSyncPacket<T>(class_2960 id, T data) implements class_8710 {
		public static final class_9154<DataSyncPacket<?>> PACKET_ID = new class_9154<>(class_2960.method_60655(CoreLib.MOD_ID, "data_sync"));

		public static final class_9139<class_2540, DataSyncPacket<?>> PACKET_CODEC = class_9139.method_56438((value, buf) -> {
			buf.method_10812(value.id);
			SyncedObject<Object> synced = get(value.id);
			synced.codecHolder().packetCodec().encode(buf, value.data);
		}, buf -> {
			class_2960 id = buf.method_10810();
			return new DataSyncPacket<>(id, get(id).codecHolder().packetCodec().decode(buf));
		});

		@Override
		public class_9154<? extends class_8710> method_56479() {
			return PACKET_ID;
		}
	}

	private Synchronisation() {}
}
